// g++ -std=c++11 compress.cpp -o compress -I ~/include -L ~/lib -lsdsl -ldivsufsort -ldivsufsort64 -O3 -DNDEBUG

#include <string>
#include <fstream>
#include <sstream>
#include <vector>


#include <sdsl/int_vector.hpp>
#include <sdsl/bit_vectors.hpp>
#include <sdsl/util.hpp>
#include <sdsl/rank_support.hpp>
#include <sdsl/select_support.hpp>
#include <sdsl/suffix_arrays.hpp>
#include <sdsl/suffix_arrays.hpp>

extern "C" {
	#include "huffman.h"
}

uint32_t compressInt(std::string filename, std::string outpath, std::string suffix);

int compressB2Huff(std::string b2file, std::string yfile, std::string outpath) {

    std::string b2fileout = outpath + ".B2.bin.huffman";
    //std::cout<<" b2fileout "<<b2fileout<<" yfile "<<yfile<<"\n";
    FILE *b2F, *b2Fout, *yF;
    if ((b2F = fopen(b2file.c_str(), "rb")) == NULL) {
	std::cout<<"error reading b2file "<<b2file<<"\n";
	exit(1);
    }
    if ((yF = fopen(yfile.c_str(), "rb")) == NULL){
	std::cout<<"error reading yfile\n";
	exit(1);
    }

    if ((b2Fout = fopen(b2fileout.c_str(), "wb")) == NULL){
	std::cout<<"error write b2Fout\n";
	exit(1);
    }

    fseek(yF,0,SEEK_END);
    int misize=ftell(yF)/4;
    int *BX = (int *)malloc(sizeof(int)*misize);
    rewind(yF);
    size_t result = fread(BX,sizeof(int),misize,yF);
    if (result != misize) {
	std::cout<<"Reading error\n"; 
	exit(1);
    }

    int status = HuffmanEncodeFileBX(b2F, b2Fout , &BX);
    
    fseek(b2Fout,0,SEEK_END);
    int b2huffsize = 8*ftell(b2Fout);

    std::string yfileHuff = yfile + ".huff.bin";
    FILE *yHuff;

    if ((yHuff = fopen(yfileHuff.c_str(), "wb")) == NULL){
	std::cout<<"error write yfileHuff\n";
	exit(1);
    }

    fwrite(BX, sizeof(int), misize, yHuff);
    fclose(yHuff);
    uint32_t yhuffsize = compressInt(yfileHuff,outpath,".Y.bin.huff.bin");
    if(status != 0){
	std::cout<<"error calling huffman\n";
	exit(1);
    }

    free(BX);
    fclose(b2F);
    fclose(b2Fout);
    fclose(yF);
    return yhuffsize + b2huffsize; 
}


uint32_t compressInt(std::string filename, std::string outpath, std::string suffix) {

    sdsl::wm_int<sdsl::rrr_vector<15>> wm_int;
    sdsl::construct(wm_int, filename.c_str(), 4);

    std::cout<<" wm file "<< outpath+suffix+"-wm_int.sdsl\n";
    store_to_file(wm_int, outpath + suffix+"-wm_int.sdsl");

    const uint32_t bitSize = size_in_bytes(wm_int)*8;

    return bitSize;
}

uint32_t compressBitmap(std::string filename, std::string outpath, std::string suffix) {

    std::ifstream Bfile (filename.c_str());

    std::string sB1;
    std::vector<uint32_t> B1;
    // B1
    if (getline (Bfile, sB1))
    {
        Bfile.close();
    }
    else
    {
        std::cerr << "Unable to open file Bfile" << std::endl;
        return -1;
    }
    std::stringstream ssB1(sB1);

    uint32_t number;
    while (ssB1 >> number)
    {
        B1.push_back(number);
    }

    size_t Blen = B1.size();

    sdsl::bit_vector B = sdsl::bit_vector(Blen, 0);
    for (size_t i=0; i <= Blen; i++) {
        if(B1[i] == 1)
        {
            B[i] = 1;
        }
    }

    uint32_t bitSize = 0;

    sdsl::rrr_vector<63> rrrb(B);
    std::cout<<" bitmap file "<< outpath+suffix+"-rrr-64.sdsl\n";
    store_to_file(rrrb, outpath + suffix+ "-rrr-64.sdsl");

    bitSize = size_in_bytes(rrrb)*8;

    return bitSize;
}


int main(int argc, char const *argv[])
{
    if(3 > argc)
    {
        std::cerr << "Modo de uso: " << argv[0] << " path/file_to_compress (extension .f or .c or .r) outdir/fileout totaledges (optional)" << std::endl;
        return -1;
    }

    std::cout << "Compressing..." << std::endl;
    // Path to files
    const std::string path(argv[1]);
    std::cout << path << std::endl;
    const std::string outpath(argv[2]);
    std::cout << outpath << std::endl;
    long totaledges = 0;
    
    if(argc==4){ // only if totaledges is given, compress will compute bpe
	totaledges = atoi(argv[3]);
    }

    std::string sXbin = path+".X.bin";
    std::string sB1 = path+".B1";
    std::string sB2bin = path+".B2.bin";
    std::string sYbin = path+".Y.bin";

    int totalSizeB2_Y = compressB2Huff(sB2bin, sYbin, outpath);
    uint32_t totalSize = 0;

    totalSize += compressInt(sXbin,outpath,".X.bin");
    // std::cout << "X Done" << std::endl;

    totalSize += compressBitmap(sB1,outpath,".B1");
    // std::cout << "B1 Done" << std::endl;

    std::cout << "Total bytes of X+B+BB+Y : " << (totalSize  + totalSizeB2_Y)/8 << std::endl;
    std::cout << "Total bits of X+B+BB+Y : " << totalSize  + totalSizeB2_Y << std::endl;

    if(totaledges > 0){
    	std::cout << "bpe : " << 1.0*(totalSize  + totalSizeB2_Y)/totaledges << std::endl;
    }


    return 0;
}
